const express = require('express');
const pool = require('../db');
const { authenticateToken } = require('./auth');
const fs = require('fs');
const path = require('path');

const router = express.Router();

// Helper: Save task change history
async function saveHistory(task_id, user_id, change_type, old_value, new_value) {
  await pool.query(
    'INSERT INTO task_history (task_id, changed_by, change_type, old_value, new_value) VALUES (?, ?, ?, ?, ?)',
    [task_id, user_id, change_type, old_value, new_value]
  );
}

// Get all categories for user
router.get('/categories', authenticateToken, async (req, res) => {
  try {
    const [categories] = await pool.query('SELECT * FROM categories WHERE user_id = ?', [req.user.id]);
    res.json(categories);
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// Add new category
router.post('/categories', authenticateToken, async (req, res) => {
  const { name } = req.body;
  if (!name) return res.status(400).json({ error: 'Category name required' });

  try {
    const [result] = await pool.query('INSERT INTO categories (user_id, name) VALUES (?, ?)', [req.user.id, name]);
    res.json({ id: result.insertId, name });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// Get all tasks (active) for user with category info
router.get('/', authenticateToken, async (req, res) => {
  try {
    const [tasks] = await pool.query(`
      SELECT t.*, c.name as category_name 
      FROM tasks t
      JOIN categories c ON t.category_id = c.id
      WHERE t.user_id = ? AND t.status = 'active'
      ORDER BY t.updated_at DESC
    `, [req.user.id]);
    res.json(tasks);
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// Create new task with optional file upload
router.post('/', authenticateToken, async (req, res) => {
  const { title, description, category_id } = req.body;
  if (!title || !category_id) return res.status(400).json({ error: 'Title and category required' });

  try {
    const [result] = await pool.query(
      'INSERT INTO tasks (user_id, category_id, title, description) VALUES (?, ?, ?, ?)',
      [req.user.id, category_id, title, description || '']
    );
    const taskId = result.insertId;

    // Handle file upload if exists
    if (req.files && req.files.file) {
      const file = req.files.file;
      const uploadDir = path.join(__dirname, '..', 'uploads');
      if (!fs.existsSync(uploadDir)) fs.mkdirSync(uploadDir);
      const filePath = path.join(uploadDir, Date.now() + '_' + file.name);
      await file.mv(filePath);

      await pool.query(
        'INSERT INTO task_files (task_id, filename, filepath) VALUES (?, ?, ?)',
        [taskId, file.name, filePath]
      );
    }

    res.json({ message: 'Task created', taskId });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// Update task (title, description, category)
router.put('/:id', authenticateToken, async (req, res) => {
  const taskId = req.params.id;
  const { title, description, category_id } = req.body;
  if (!title || !category_id) return res.status(400).json({ error: 'Title and category required' });

  try {
    const [oldRows] = await pool.query('SELECT * FROM tasks WHERE id = ? AND user_id = ?', [taskId, req.user.id]);
    if (oldRows.length === 0) return res.status(404).json({ error: 'Task not found' });

    const oldTask = oldRows[0];

    await pool.query(
      'UPDATE tasks SET title = ?, description = ?, category_id = ? WHERE id = ? AND user_id = ?',
      [title, description || '', category_id, taskId, req.user.id]
    );

    await saveHistory(taskId, req.user.id, 'edit', JSON.stringify(oldTask), JSON.stringify({ title, description, category_id }));

    res.json({ message: 'Task updated' });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// Soft delete task (move to trash)
router.delete('/:id', authenticateToken, async (req, res) => {
  const taskId = req.params.id;

  try {
    const [rows] = await pool.query('SELECT * FROM tasks WHERE id = ? AND user_id = ?', [taskId, req.user.id]);
    if (rows.length === 0) return res.status(404).json({ error: 'Task not found' });

    await pool.query("UPDATE tasks SET status = 'deleted' WHERE id = ? AND user_id = ?", [taskId, req.user.id]);

    await saveHistory(taskId, req.user.id, 'delete', JSON.stringify(rows[0]), null);

    res.json({ message: 'Task moved to trash' });
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

// Get trash tasks
router.get('/trash', authenticateToken, async (req, res) => {
  try {
    const [tasks] = await pool.query('SELECT * FROM tasks WHERE user_id = ? AND status = "deleted" ORDER BY updated_at DESC', [req.user.id]);
    res.json(tasks);
  } catch (e) {
    res.status(500).json({ error: e.message });
  }
});

module.exports = router;
